<?php
/**
 * @author     Kristof Ringleff
 * @package    Fooman_PdfCustomiser
 * @copyright  Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\PdfCustomiser\Block;

class Totals extends \Fooman\PdfCore\Block\Pdf\Block
{
    protected $_template = 'Fooman_PdfCustomiser::pdf/totals.phtml';
    private $totalsHelper;
    private $taxConfig;

    /**
     * @param \Magento\Framework\View\Element\Template\Context $context
     * @param \Fooman\PdfCustomiser\Helper\Totals              $totalsHelper
     * @param \Magento\Tax\Model\Config                        $taxConfig
     * @param array                                            $data
     */
    public function __construct(
        \Magento\Framework\View\Element\Template\Context $context,
        \Fooman\PdfCustomiser\Helper\Totals $totalsHelper,
        \Magento\Tax\Model\Config $taxConfig,
        array $data = []
    ) {
        $this->totalsHelper = $totalsHelper;
        $this->taxConfig = $taxConfig;
        parent::__construct($context, $data);
    }

    /**
     * @return \Magento\Sales\Model\Order\Pdf\Total\DefaultTotal[]
     */
    public function getTotals()
    {
        $totals = $this->totalsHelper->getTotalsList();
        $filterTotals = [];
        if ($this->getSalesObject() instanceof \Magento\Sales\Api\Data\CreditmemoInterface) {
            return $totals;
        }
        foreach ($totals as $total) {
            if ($total->getSourceField() !== 'adjustment_positive'
                && $total->getSourceField() !== 'adjustment_negative') {
                $filterTotals[] = $total;
            }
        }
        return $filterTotals;
    }

    /**
     * @param \Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total
     *
     * @return \Magento\Sales\Model\Order\Pdf\Total\DefaultTotal
     */
    public function prepareTotal(\Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total)
    {
        $total->setOrder($this->getOrder());
        $total->setSource($this->getSalesObject());
        $this->getSalesObject()->setOrder($this->getOrder());
        return $total;
    }

    public function getTotalLinesForDisplay(\Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total)
    {
        $totalLines = $total->getTotalsForDisplay();
        if ($this->isTaxDisplayedWithGrandTotal($total)) {
            $fullTaxInfo = $total->getFullTaxInfo();
            $grandTotalExl = array_shift($totalLines);
            $baseAmount = $this->getSalesObject()->getBaseGrandTotal();
            $baseTaxAmount = $this->getSalesObject()->getBaseTaxAmount();

            $grandTotalExl['base_amount'] = $this->getAmountPrefix()
                . $this->getOrder()->getBaseCurrency()->formatTxt($baseAmount - $baseTaxAmount);

            $grandTotalIncl = array_pop($totalLines);
            $grandTotalIncl['base_amount'] = $this->getAmountPrefix()
                . $this->getOrder()->getBaseCurrency()->formatTxt($baseAmount);

            $totalLines = [$grandTotalExl];
            if ($baseTaxAmount > 0) {
                foreach ($fullTaxInfo as $taxLine) {
                    $line = [];
                    $line['label'] = $taxLine['title'] . ' (' . (float)$taxLine['percent'] . '%):';
                    $line['amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getOrderCurrency()->formatTxt($taxLine['tax_amount']);
                    $line['base_amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getBaseCurrency()->formatTxt($taxLine['base_tax_amount']);
                    $totalLines[] = $line;
                }
            } else {
                // should be 0 - due to an issue with Magento it will wrongly retrieve the tax rates from the order
                // in case we are displaying a credit memo with no items
                // we loop over the rates to get the label
                foreach ($fullTaxInfo as $taxLine) {
                    $line = [];
                    $line['label'] = $taxLine['label'];
                    $line['amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getOrderCurrency()->formatTxt($this->getSalesObject()->getTaxAmount());
                    $line['base_amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getBaseCurrency()->formatTxt($this->getSalesObject()->getBaseTaxAmount());
                    $totalLines[] = $line;
                }
            }

            $totalLines[] = $grandTotalIncl;
        } elseif ($this->isFullTaxDisplayed($total)) {
            $fullTaxInfo = $total->getFullTaxInfo();
            $totalLines = [];
            if ($this->getSalesObject()->getBaseTaxAmount() > 0) {
                foreach ($fullTaxInfo as $taxLine) {
                    $line = [];
                    $line['label'] = $taxLine['title'] . ' (' . (float)$taxLine['percent'] . '%):';
                    $line['amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getOrderCurrency()->formatTxt($taxLine['tax_amount']);
                    $line['base_amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getBaseCurrency()->formatTxt($taxLine['base_tax_amount']);
                    $totalLines[] = $line;
                }
            } else {
                // should be 0 - due to an issue with Magento it will wrongly retrieve the tax rates from the order
                // in case we are displaying a credit memo with no items
                // we loop over the rates to get the label
                foreach ($fullTaxInfo as $taxLine) {
                    $line = [];
                    $line['label'] = $taxLine['title'] . ' (' . (float)$taxLine['percent'] . '%):';
                    $line['amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getOrderCurrency()->formatTxt($this->getSalesObject()->getTaxAmount());
                    $line['base_amount'] = $this->getAmountPrefix()
                        . $this->getOrder()->getBaseCurrency()->formatTxt($this->getSalesObject()->getBaseTaxAmount());
                    $totalLines[] = $line;
                }
            }
        } else {
            foreach ($totalLines as $key => $line) {
                $baseAmount = $this->getSalesObject()->getDataUsingMethod('base_' . $total->getSourceField());
                $line['base_amount'] = $this->getAmountPrefix()
                    . $this->getOrder()->getBaseCurrency()->formatTxt($baseAmount);

                $totalLines[$key] = $line;
            }
        }

        return $totalLines;
    }

    public function shouldDisplayBothCurrencies()
    {
        $enabled = $this->_scopeConfig->getValue(
            AbstractSalesDocument::XML_PATH_DISPLAYBOTH,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $this->getOrder()->getStoreId()
        );

        return $enabled
            && $this->getOrder()->getBaseCurrencyCode() !== $this->getOrder()->getOrderCurrencyCode();
    }

    /**
     * @param \Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total
     *
     * @return bool
     */
    private function isTaxDisplayedWithGrandTotal(\Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total)
    {
        return $total->getSourceField() === 'grand_total' &&
            $this->taxConfig->displaySalesTaxWithGrandTotal(
                $this->getOrder()->getStoreId()
            );
    }

    /**
     * @param \Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total
     *
     * @return bool
     */
    private function isFullTaxDisplayed(\Magento\Sales\Model\Order\Pdf\Total\DefaultTotal $total)
    {
        return $total->getSourceField() === 'tax_amount' &&
            $this->taxConfig->displaySalesFullSummary(
                $this->getOrder()->getStoreId()
            ) && !$this->taxConfig->displaySalesTaxWithGrandTotal(
                $this->getOrder()->getStoreId()
            );
    }
}
